package ca.training.bigdata.yarn.demo;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.hadoop.yarn.api.ApplicationConstants;
import org.apache.hadoop.yarn.api.records.Container;
import org.apache.hadoop.yarn.api.records.ContainerLaunchContext;
import org.apache.hadoop.yarn.api.records.ContainerStatus;
import org.apache.hadoop.yarn.api.records.FinalApplicationStatus;
import org.apache.hadoop.yarn.api.records.LocalResource;
import org.apache.hadoop.yarn.api.records.NodeReport;
import org.apache.hadoop.yarn.api.records.Priority;
import org.apache.hadoop.yarn.api.records.Resource;
import org.apache.hadoop.yarn.client.api.AMRMClient.ContainerRequest;
import org.apache.hadoop.yarn.client.api.NMClient;
import org.apache.hadoop.yarn.client.api.async.AMRMClientAsync;
import org.apache.hadoop.yarn.conf.YarnConfiguration;
import org.apache.hadoop.yarn.util.Records;

public class DemoAppMaster implements AMRMClientAsync.CallbackHandler {

	private YarnConfiguration conf = new YarnConfiguration();
	private NMClient nmClient;
	private int containerCount = 3;

	public static void main(String[] args) {
		System.out.println("AppMaster: Initializing");
		try {
			new DemoAppMaster().run();
		} catch (Exception ex) {
			ex.printStackTrace();
		}
	}

	public void run() throws Exception {
		conf = new YarnConfiguration();

		nmClient = NMClient.createNMClient();
		nmClient.init(conf);
		nmClient.start();
		
		AMRMClientAsync<ContainerRequest> rmClient = AMRMClientAsync.createAMRMClientAsync(1000, this);
		rmClient.init(conf);
		rmClient.start();
		
		rmClient.registerApplicationMaster("", 0, "");
		System.out.println("AppMaster: Registered");

		Priority priority = Records.newRecord(Priority.class);
		priority.setPriority(0);
		
		Resource capability = Records.newRecord(Resource.class);
		capability.setMemory(128);
		capability.setVirtualCores(1);

		System.out.println("AppMaster: Requesting " + containerCount + " Containers");
		for (int i = 0; i < containerCount; ++i) {
			rmClient.addContainerRequest(new ContainerRequest(capability, null, null, priority));
		}
		
		while (!containersFinished()) {
			Thread.sleep(100);
		}

		System.out.println("AppMaster: Unregistered");
		rmClient.unregisterApplicationMaster(FinalApplicationStatus.SUCCEEDED, "", "");
	}

	private boolean containersFinished() {
		return containerCount == 0;
	}

	@Override
	public void onContainersAllocated(List<Container> containers) {
		for (Container container : containers) {
			try {
				nmClient.startContainer(container, initContainer());
				System.err.println("AppMaster: Container launched " + container.getId());
			} catch (Exception ex) {
				System.err.println("AppMaster: Container not launched " + container.getId());
				ex.printStackTrace();
			}
		}
	}

	private ContainerLaunchContext initContainer() {
		try {
			ContainerLaunchContext cCLC = Records.newRecord(ContainerLaunchContext.class);
			cCLC.setCommands(Collections.singletonList("$JAVA_HOME/bin/java"
				+ " -Xmx256M"
				+ " ca.training.bigdata.yarn.demo.DemoContainer"
				+ " 1>" + ApplicationConstants.LOG_DIR_EXPANSION_VAR + "/stdout"
				+ " 2>" + ApplicationConstants.LOG_DIR_EXPANSION_VAR + "/stderr"));

			LocalResource jar = Records.newRecord(LocalResource.class);
			DemoUtils.setUpLocalResource(DemoUtils.YARNAPP_JAR_PATH, jar, conf);
			cCLC.setLocalResources(Collections.singletonMap(DemoUtils.YARNAPP_JAR_NAME, jar));
			
			Map<String, String> env = new HashMap<String, String>();
			DemoUtils.setUpEnv(env, conf);
			cCLC.setEnvironment(env);

			return cCLC;
		} catch (Exception ex) {
			ex.printStackTrace();
			return null;
		}
	}

	@Override
	public void onContainersCompleted(List<ContainerStatus> statusOfContainers) {
		for (ContainerStatus status : statusOfContainers) {
			System.err.println("AppMaster: Container finished " + status.getContainerId());
			synchronized (this) {
				containerCount--;
			}
		}
	}

	@Override
	public void onError(Throwable e) {}

	@Override
	public void onNodesUpdated(List<NodeReport> nodeReports) {}

	@Override
	public void onShutdownRequest() {}

	@Override
	public float getProgress() {
		return 0;
	}
}
