package ca.training.bigdata.spark.streaming.receiver

import java.io.{BufferedReader, InputStreamReader}
import java.net.Socket
import java.nio.charset.StandardCharsets

import org.apache.spark.SparkConf
import org.apache.spark.internal.Logging
import org.apache.spark.storage.StorageLevel
import org.apache.spark.streaming.{Seconds, StreamingContext}
import org.apache.spark.streaming.receiver.Receiver

/**
  * Created by BigDataTraining on 2018-03-16.
  */
object CustomReceiver {

    def main(args: Array[String]) {
      if (args.length < 2) {
        System.err.println("Usage: CustomReceiver <hostname> <port>")
        System.exit(1)
      }

      // Create the context with a 1 second batch size
      val sparkConf = new SparkConf().setAppName("CustomReceiver")
      val ssc = new StreamingContext(sparkConf, Seconds(10))

      // Create an input stream with the custom receiver on target ip:port and count the
      // words in input stream of \n delimited text (eg. generated by 'nc')
      val lines = ssc.receiverStream(new CustomReceiver(args(0), args(1).toInt))
      val words = lines.flatMap(_.split(" "))
      val wordCounts = words.map(x => (x, 1)).reduceByKey(_ + _)
      wordCounts.print()
      ssc.start()
      ssc.awaitTermination()
    }
  }


  class CustomReceiver(host: String, port: Int)
    extends Receiver[String](StorageLevel.MEMORY_AND_DISK_2) {

    def onStart() {
      // Start the thread that receives data over a connection
      new Thread("Socket Receiver") {
        override def run() { receive() }
      }.start()
    }

    def onStop() {
    }

    /** Create a socket connection and receive data until receiver is stopped */
    private def receive() {
      var socket: Socket = null
      var userInput: String = null
      try {
        println("Connecting to " + host + ":" + port)
        socket = new Socket(host, port)
        println("Connected to " + host + ":" + port)
        val reader = new BufferedReader(
          new InputStreamReader(socket.getInputStream(), StandardCharsets.UTF_8))
        userInput = reader.readLine()
        while(!isStopped && userInput != null) {
          store(userInput)
          userInput = reader.readLine()
        }
        reader.close()
        socket.close()
        println("Stopped receiving")
        restart("Trying to connect again")
      } catch {
        case e: java.net.ConnectException =>
          restart("Error connecting to " + host + ":" + port, e)
        case t: Throwable =>
          restart("Error receiving data", t)
      }
    }

}
