package ca.training.bigdata.kafka.streaming

import org.apache.kafka.common.serialization.StringDeserializer
import org.apache.spark.SparkConf
import org.apache.spark.sql.SparkSession
import org.apache.spark.streaming.{Seconds, StreamingContext}
import org.apache.spark.streaming.kafka010._
import org.apache.spark.streaming.kafka010.ConsumerStrategies.Subscribe
import org.apache.spark.streaming.kafka010.LocationStrategies.PreferConsistent

object SparkStreamingApp {

  def main(args: Array[String]): Unit = {

    val sparkConf = new SparkConf().setAppName("Move data from kafka to Hive and Elastic Search using Spark Streaming")
    val ssc = new StreamingContext(sparkConf, Seconds(30))

    val kafkaParams = Map[String, Object](
      "bootstrap.servers" -> "sandbox-hdp.hortonworks.com:6667",
      "key.deserializer" -> classOf[StringDeserializer],
      "value.deserializer" -> classOf[StringDeserializer],
      "group.id" -> "streamingGroup",
      "auto.offset.reset" -> "latest",
      "enable.auto.commit" -> (false: java.lang.Boolean)
    )

    val elasticConf: Map[String, String] = Map("es.nodes" -> "localhost",
      "es.port" -> "9200",
      "es.nodes.wan.only" -> "true",
      "es.mapping.date.rich" -> "false",
      "pushdown" -> "true")

    val topics = Array("good-topic")

    val stream = KafkaUtils.createDirectStream[String, String](
      ssc,
      PreferConsistent,
      Subscribe[String, String](topics, kafkaParams)
    )

    stream.map(record => record.value).foreachRDD(rdd => {
      val spark = SparkSession.builder.config(rdd.sparkContext.getConf).enableHiveSupport().getOrCreate()
      if (!rdd.isEmpty()) {
        val df = spark.read.json(rdd)
        df.cache()
        df.write.mode("append").format("parquet").saveAsTable("kafka.customer_bitcoin")
        df.write.mode("append").format("es").options(elasticConf).save("kafka-streaming-index/bitcoin")
      }

    })

    ssc.start()
    ssc.awaitTermination()
  }

}
