/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.jdbc.source;

import io.confluent.connect.jdbc.source.BulkTableQuerier;
import io.confluent.connect.jdbc.source.JdbcSourceTaskConfig;
import io.confluent.connect.jdbc.source.TableQuerier;
import io.confluent.connect.jdbc.source.TimestampIncrementingTableQuerier;
import io.confluent.connect.jdbc.util.CachedConnectionProvider;
import io.confluent.connect.jdbc.util.JdbcUtils;
import io.confluent.connect.jdbc.util.Version;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.config.types.Password;
import org.apache.kafka.common.utils.SystemTime;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.source.SourceRecord;
import org.apache.kafka.connect.source.SourceTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JdbcSourceTask
extends SourceTask {
    private static final Logger log = LoggerFactory.getLogger(JdbcSourceTask.class);
    private Time time;
    private JdbcSourceTaskConfig config;
    private CachedConnectionProvider cachedConnectionProvider;
    private PriorityQueue<TableQuerier> tableQueue = new PriorityQueue();
    private AtomicBoolean stop;

    public JdbcSourceTask() {
        this.time = new SystemTime();
    }

    public JdbcSourceTask(Time time) {
        this.time = time;
    }

    public String version() {
        return Version.getVersion();
    }

    public void start(Map<String, String> properties) {
        try {
            this.config = new JdbcSourceTaskConfig(properties);
        }
        catch (ConfigException e) {
            throw new ConnectException("Couldn't start JdbcSourceTask due to configuration error", (Throwable)e);
        }
        this.createConnectionProvider();
        List<String> tables = this.config.getList("tables");
        String query = this.config.getString("query");
        if (tables.isEmpty() && query.isEmpty() || !tables.isEmpty() && !query.isEmpty()) {
            throw new ConnectException("Invalid configuration: each JdbcSourceTask must have at least one table assigned to it or one query specified");
        }
        TableQuerier.QueryMode queryMode = !query.isEmpty() ? TableQuerier.QueryMode.QUERY : TableQuerier.QueryMode.TABLE;
        List<String> tablesOrQuery = queryMode == TableQuerier.QueryMode.QUERY ? Collections.singletonList(query) : tables;
        String mode = this.config.getString("mode");
        Map offsets = null;
        if (mode.equals("incrementing") || mode.equals("timestamp") || mode.equals("timestamp+incrementing")) {
            ArrayList<Map<String, String>> partitions = new ArrayList<Map<String, String>>(tables.size());
            switch (queryMode) {
                case TABLE: {
                    for (String table : tables) {
                        Map<String, String> partition = Collections.singletonMap("table", table);
                        partitions.add(partition);
                    }
                    break;
                }
                case QUERY: {
                    partitions.add(Collections.singletonMap("query", "query"));
                    break;
                }
                default: {
                    throw new ConnectException("Unknown query mode: " + (Object)((Object)queryMode));
                }
            }
            offsets = this.context.offsetStorageReader().offsets(partitions);
        }
        String schemaPattern = this.config.getString("schema.pattern");
        String incrementingColumn = this.config.getString("incrementing.column.name");
        String timestampColumn = this.config.getString("timestamp.column.name");
        Long timestampDelayInterval = this.config.getLong("timestamp.delay.interval.ms");
        boolean validateNonNulls = this.config.getBoolean("validate.non.null");
        for (String tableOrQuery : tablesOrQuery) {
            Map<String, String> partition;
            switch (queryMode) {
                case TABLE: {
                    if (validateNonNulls) {
                        this.validateNonNullable(mode, schemaPattern, tableOrQuery, incrementingColumn, timestampColumn);
                    }
                    partition = Collections.singletonMap("table", tableOrQuery);
                    break;
                }
                case QUERY: {
                    partition = Collections.singletonMap("query", "query");
                    break;
                }
                default: {
                    throw new ConnectException("Unexpected query mode: " + (Object)((Object)queryMode));
                }
            }
            Map offset = offsets == null ? null : (Map)offsets.get(partition);
            String topicPrefix = this.config.getString("topic.prefix");
            boolean mapNumerics = this.config.getBoolean("numeric.precision.mapping");
            if (mode.equals("bulk")) {
                this.tableQueue.add(new BulkTableQuerier(queryMode, tableOrQuery, schemaPattern, topicPrefix, mapNumerics));
                continue;
            }
            if (mode.equals("incrementing")) {
                this.tableQueue.add(new TimestampIncrementingTableQuerier(queryMode, tableOrQuery, topicPrefix, null, incrementingColumn, offset, timestampDelayInterval, schemaPattern, mapNumerics));
                continue;
            }
            if (mode.equals("timestamp")) {
                this.tableQueue.add(new TimestampIncrementingTableQuerier(queryMode, tableOrQuery, topicPrefix, timestampColumn, null, offset, timestampDelayInterval, schemaPattern, mapNumerics));
                continue;
            }
            if (!mode.endsWith("timestamp+incrementing")) continue;
            this.tableQueue.add(new TimestampIncrementingTableQuerier(queryMode, tableOrQuery, topicPrefix, timestampColumn, incrementingColumn, offset, timestampDelayInterval, schemaPattern, mapNumerics));
        }
        this.stop = new AtomicBoolean(false);
    }

    private void createConnectionProvider() {
        String dbUrl = this.config.getString("connection.url");
        String dbUser = this.config.getString("connection.user");
        Password dbPassword = this.config.getPassword("connection.password");
        int maxConnectionAttempts = this.config.getInt("connection.attempts");
        long connectionRetryBackoff = this.config.getLong("connection.backoff.ms");
        this.cachedConnectionProvider = new CachedConnectionProvider(dbUrl, dbUser, dbPassword == null ? null : dbPassword.value(), maxConnectionAttempts, connectionRetryBackoff);
    }

    public void stop() throws ConnectException {
        if (this.stop != null) {
            this.stop.set(true);
        }
        if (this.cachedConnectionProvider != null) {
            this.cachedConnectionProvider.closeQuietly();
        }
    }

    public List<SourceRecord> poll() throws InterruptedException {
        log.trace("{} Polling for new data");
        while (!this.stop.get()) {
            long nextUpdate;
            long untilNext;
            TableQuerier querier = this.tableQueue.peek();
            if (!querier.querying() && (untilNext = (nextUpdate = querier.getLastUpdate() + (long)this.config.getInt("poll.interval.ms").intValue()) - this.time.milliseconds()) > 0L) {
                log.trace("Waiting {} ms to poll {} next", (Object)untilNext, (Object)querier.toString());
                this.time.sleep(untilNext);
                continue;
            }
            ArrayList<SourceRecord> results = new ArrayList<SourceRecord>();
            try {
                log.debug("Checking for next block of results from {}", (Object)querier.toString());
                querier.maybeStartQuery(this.cachedConnectionProvider.getValidConnection());
                int batchMaxRows = this.config.getInt("batch.max.rows");
                boolean hadNext = true;
                while (results.size() < batchMaxRows && (hadNext = querier.next())) {
                    results.add(querier.extractRecord());
                }
                if (!hadNext) {
                    this.resetAndRequeueHead(querier);
                }
                if (results.isEmpty()) {
                    log.trace("No updates for {}", (Object)querier.toString());
                    continue;
                }
                log.debug("Returning {} records for {}", (Object)results.size(), (Object)querier.toString());
                return results;
            }
            catch (SQLException e) {
                log.error("Failed to run query for table {}: {}", (Object)querier.toString(), (Object)e);
                this.resetAndRequeueHead(querier);
                return null;
            }
        }
        return null;
    }

    private void resetAndRequeueHead(TableQuerier expectedHead) {
        log.debug("Resetting querier {}", (Object)expectedHead.toString());
        TableQuerier removedQuerier = this.tableQueue.poll();
        assert (removedQuerier == expectedHead);
        expectedHead.reset(this.time.milliseconds());
        this.tableQueue.add(expectedHead);
    }

    private void validateNonNullable(String incrementalMode, String schemaPattern, String table, String incrementingColumn, String timestampColumn) {
        try {
            Connection connection = this.cachedConnectionProvider.getValidConnection();
            if ((incrementalMode.equals("incrementing") || incrementalMode.equals("timestamp+incrementing")) && JdbcUtils.isColumnNullable(connection, schemaPattern, table, incrementingColumn)) {
                throw new ConnectException("Cannot make incremental queries using incrementing column " + incrementingColumn + " on " + table + " because this column " + "is nullable.");
            }
            if ((incrementalMode.equals("timestamp") || incrementalMode.equals("timestamp+incrementing")) && JdbcUtils.isColumnNullable(connection, schemaPattern, table, timestampColumn)) {
                throw new ConnectException("Cannot make incremental queries using timestamp column " + timestampColumn + " on " + table + " because this column is " + "nullable.");
            }
        }
        catch (SQLException e) {
            throw new ConnectException("Failed trying to validate that columns used for offsets are NOT NULL", (Throwable)e);
        }
    }
}

