/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.jdbc.sink.dialect;

import io.confluent.connect.jdbc.sink.dialect.GenericDialect;
import io.confluent.connect.jdbc.sink.dialect.HanaDialect;
import io.confluent.connect.jdbc.sink.dialect.MySqlDialect;
import io.confluent.connect.jdbc.sink.dialect.OracleDialect;
import io.confluent.connect.jdbc.sink.dialect.PostgreSqlDialect;
import io.confluent.connect.jdbc.sink.dialect.SqlServerDialect;
import io.confluent.connect.jdbc.sink.dialect.SqliteDialect;
import io.confluent.connect.jdbc.sink.dialect.StringBuilderUtil;
import io.confluent.connect.jdbc.sink.dialect.VerticaDialect;
import io.confluent.connect.jdbc.sink.metadata.SinkRecordField;
import io.confluent.connect.jdbc.util.DateTimeUtils;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import javax.xml.bind.DatatypeConverter;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.errors.ConnectException;

public abstract class DbDialect {
    private final String escapeStart;
    private final String escapeEnd;

    DbDialect(String escapeStart, String escapeEnd) {
        this.escapeStart = escapeStart;
        this.escapeEnd = escapeEnd;
    }

    public final String getInsert(String tableName, Collection<String> keyColumns, Collection<String> nonKeyColumns) {
        StringBuilder builder = new StringBuilder("INSERT INTO ");
        builder.append(this.escaped(tableName));
        builder.append("(");
        StringBuilderUtil.joinToBuilder(builder, ",", keyColumns, nonKeyColumns, this.escaper());
        builder.append(") VALUES(");
        StringBuilderUtil.copiesToBuilder(builder, ",", "?", keyColumns.size() + nonKeyColumns.size());
        builder.append(")");
        return builder.toString();
    }

    public final String getUpdate(String tableName, Collection<String> keyColumns, Collection<String> nonKeyColumns) {
        StringBuilder builder = new StringBuilder("UPDATE ");
        builder.append(this.escaped(tableName));
        builder.append(" SET ");
        StringBuilderUtil.Transform<String> updateTransformer = new StringBuilderUtil.Transform<String>(){

            @Override
            public void apply(StringBuilder builder, String input) {
                builder.append(DbDialect.this.escaped(input));
                builder.append(" = ?");
            }
        };
        StringBuilderUtil.joinToBuilder(builder, ", ", nonKeyColumns, updateTransformer);
        if (!keyColumns.isEmpty()) {
            builder.append(" WHERE ");
        }
        StringBuilderUtil.joinToBuilder(builder, ", ", keyColumns, updateTransformer);
        return builder.toString();
    }

    public String getUpsertQuery(String table, Collection<String> keyColumns, Collection<String> columns) {
        throw new UnsupportedOperationException();
    }

    public String getCreateQuery(String tableName, Collection<SinkRecordField> fields) {
        List<String> pkFieldNames = DbDialect.extractPrimaryKeyFieldNames(fields);
        StringBuilder builder = new StringBuilder();
        builder.append("CREATE TABLE ");
        builder.append(this.escaped(tableName));
        builder.append(" (");
        this.writeColumnsSpec(builder, fields);
        if (!pkFieldNames.isEmpty()) {
            builder.append(",");
            builder.append(System.lineSeparator());
            builder.append("PRIMARY KEY(");
            StringBuilderUtil.joinToBuilder(builder, ",", pkFieldNames, this.escaper());
            builder.append(")");
        }
        builder.append(")");
        return builder.toString();
    }

    public List<String> getAlterTable(String tableName, Collection<SinkRecordField> fields) {
        final boolean newlines = fields.size() > 1;
        StringBuilder builder = new StringBuilder("ALTER TABLE ");
        builder.append(this.escaped(tableName));
        builder.append(" ");
        StringBuilderUtil.joinToBuilder(builder, ",", fields, new StringBuilderUtil.Transform<SinkRecordField>(){

            @Override
            public void apply(StringBuilder builder, SinkRecordField f) {
                if (newlines) {
                    builder.append(System.lineSeparator());
                }
                builder.append("ADD ");
                DbDialect.this.writeColumnSpec(builder, f);
            }
        });
        return Collections.singletonList(builder.toString());
    }

    protected void writeColumnsSpec(StringBuilder builder, Collection<SinkRecordField> fields) {
        StringBuilderUtil.joinToBuilder(builder, ",", fields, new StringBuilderUtil.Transform<SinkRecordField>(){

            @Override
            public void apply(StringBuilder builder, SinkRecordField f) {
                builder.append(System.lineSeparator());
                DbDialect.this.writeColumnSpec(builder, f);
            }
        });
    }

    protected void writeColumnSpec(StringBuilder builder, SinkRecordField f) {
        builder.append(this.escaped(f.name()));
        builder.append(" ");
        builder.append(this.getSqlType(f.schemaName(), f.schemaParameters(), f.schemaType()));
        if (f.defaultValue() != null) {
            builder.append(" DEFAULT ");
            this.formatColumnValue(builder, f.schemaName(), f.schemaParameters(), f.schemaType(), f.defaultValue());
        } else if (f.isOptional()) {
            builder.append(" NULL");
        } else {
            builder.append(" NOT NULL");
        }
    }

    protected void formatColumnValue(StringBuilder builder, String schemaName, Map<String, String> schemaParameters, Schema.Type type, Object value) {
        if (schemaName != null) {
            switch (schemaName) {
                case "org.apache.kafka.connect.data.Decimal": {
                    builder.append(value);
                    return;
                }
                case "org.apache.kafka.connect.data.Date": {
                    builder.append("'").append(DateTimeUtils.formatUtcDate((Date)value)).append("'");
                    return;
                }
                case "org.apache.kafka.connect.data.Time": {
                    builder.append("'").append(DateTimeUtils.formatUtcTime((Date)value)).append("'");
                    return;
                }
                case "org.apache.kafka.connect.data.Timestamp": {
                    builder.append("'").append(DateTimeUtils.formatUtcTimestamp((Date)value)).append("'");
                    return;
                }
            }
        }
        switch (type) {
            case INT8: 
            case INT16: 
            case INT32: 
            case INT64: 
            case FLOAT32: 
            case FLOAT64: {
                builder.append(value);
                break;
            }
            case BOOLEAN: {
                builder.append((Boolean)value != false ? (char)'1' : '0');
                break;
            }
            case STRING: {
                builder.append("'").append(value).append("'");
                break;
            }
            case BYTES: {
                byte[] bytes;
                if (value instanceof ByteBuffer) {
                    ByteBuffer buffer = ((ByteBuffer)value).slice();
                    bytes = new byte[buffer.remaining()];
                    buffer.get(bytes);
                } else {
                    bytes = (byte[])value;
                }
                builder.append("x'").append(DatatypeConverter.printHexBinary((byte[])bytes)).append("'");
                break;
            }
            default: {
                throw new ConnectException("Unsupported type for column value: " + type);
            }
        }
    }

    protected String getSqlType(String schemaName, Map<String, String> parameters, Schema.Type type) {
        throw new ConnectException(String.format("%s (%s) type doesn't have a mapping to the SQL database column type", schemaName, type));
    }

    protected String escaped(String identifier) {
        return this.escapeStart + identifier + this.escapeEnd;
    }

    protected StringBuilderUtil.Transform<String> escaper() {
        return new StringBuilderUtil.Transform<String>(){

            @Override
            public void apply(StringBuilder builder, String identifier) {
                builder.append(DbDialect.this.escapeStart).append(identifier).append(DbDialect.this.escapeEnd);
            }
        };
    }

    protected StringBuilderUtil.Transform<String> prefixedEscaper(final String prefix) {
        return new StringBuilderUtil.Transform<String>(){

            @Override
            public void apply(StringBuilder builder, String identifier) {
                builder.append(prefix).append(DbDialect.this.escapeStart).append(identifier).append(DbDialect.this.escapeEnd);
            }
        };
    }

    static List<String> extractPrimaryKeyFieldNames(Collection<SinkRecordField> fields) {
        ArrayList<String> pks = new ArrayList<String>();
        for (SinkRecordField f : fields) {
            if (!f.isPrimaryKey()) continue;
            pks.add(f.name());
        }
        return pks;
    }

    public static DbDialect fromConnectionString(String url) {
        String protocol;
        if (!url.startsWith("jdbc:")) {
            throw new ConnectException(String.format("Not a valid JDBC URL: %s", url));
        }
        if (url.startsWith("jdbc:sqlite:")) {
            return new SqliteDialect();
        }
        if (url.startsWith("jdbc:oracle:thin:@")) {
            return new OracleDialect();
        }
        if (url.startsWith("jdbc:sap")) {
            return new HanaDialect();
        }
        if (url.startsWith("jdbc:vertica")) {
            return new VerticaDialect();
        }
        switch (protocol = DbDialect.extractProtocolFromUrl(url).toLowerCase()) {
            case "microsoft:sqlserver": 
            case "sqlserver": 
            case "jtds:sqlserver": {
                return new SqlServerDialect();
            }
            case "mariadb": 
            case "mysql": {
                return new MySqlDialect();
            }
            case "postgresql": {
                return new PostgreSqlDialect();
            }
        }
        return new GenericDialect();
    }

    static String extractProtocolFromUrl(String url) {
        if (!url.startsWith("jdbc:")) {
            throw new ConnectException(String.format("Not a valid JDBC URL: %s", url));
        }
        int index = url.indexOf("://", "jdbc:".length());
        if (index < 0) {
            throw new ConnectException(String.format("Not a valid JDBC URL: %s", url));
        }
        return url.substring("jdbc:".length(), index);
    }
}

