/*
 * Decompiled with CFR 0.152.
 */
package hivemall.utils.buffer;

import hivemall.utils.lang.NumberUtils;
import hivemall.utils.lang.Preconditions;
import hivemall.utils.lang.Primitives;
import hivemall.utils.lang.UnsafeUtils;
import javax.annotation.Nonnull;
import javax.annotation.concurrent.NotThreadSafe;
import sun.misc.Unsafe;

@NotThreadSafe
public final class HeapBuffer {
    public static final int DEFAULT_CHUNK_SIZE = 0x400000;
    public static final int DEFAULT_CHUNK_BYTES = 0x1000000;
    public static final int DEFAULT_NUM_CHUNKS = 8;
    @Nonnull
    private final Unsafe _UNSAFE = UnsafeUtils.getUnsafe();
    private final int _chunkSize;
    private final int _chunkBytes;
    @Nonnull
    private int[][] _chunks;
    private int _initializedChunks;
    private long _position;
    private int _numAllocated;
    private long _allocatedBytes;
    private long _skippedBytes;

    public HeapBuffer() {
        this(0x400000);
    }

    public HeapBuffer(int chunkSize) {
        this(chunkSize, 8);
    }

    public int getChunkSize() {
        return this._chunkBytes;
    }

    public HeapBuffer(int chunkSize, int initNumChunks) {
        this._chunkSize = chunkSize;
        this._chunkBytes = 4 * chunkSize;
        this._chunks = new int[initNumChunks][];
        this._initializedChunks = 0;
        this._position = 0L;
        this._numAllocated = 0;
        this._allocatedBytes = 0L;
        this._skippedBytes = 0L;
    }

    public long allocate(int bytes) {
        Preconditions.checkArgument(bytes > 0, "Failed to allocate bytes : %s", bytes);
        Preconditions.checkArgument(bytes <= this._chunkBytes, "Cannot allocate memory greater than %s bytes: %s", this._chunkBytes, bytes);
        int i = Primitives.castToInt(this._position / (long)this._chunkBytes);
        int j = Primitives.castToInt(this._position % (long)this._chunkBytes);
        if (bytes > this._chunkBytes - j) {
            this._skippedBytes += (long)(this._chunkBytes - j);
            this._position = (long)(++i) * (long)this._chunkBytes;
        }
        this.grow(i);
        long ptr = this._position;
        this._position += (long)bytes;
        this._allocatedBytes += (long)bytes;
        ++this._numAllocated;
        return ptr;
    }

    private void grow(int chunkIndex) {
        if (chunkIndex < this._initializedChunks) {
            return;
        }
        Object chunks = this._chunks;
        if (chunkIndex >= this._chunks.length) {
            int newSize = Math.max(chunkIndex + 1, this._chunks.length * 2);
            int[][] newChunks = new int[newSize][];
            System.arraycopy(this._chunks, 0, newChunks, 0, this._chunks.length);
            this._chunks = newChunks;
            chunks = newChunks;
        }
        for (int i = this._initializedChunks; i <= chunkIndex; ++i) {
            chunks[i] = new int[this._chunkSize];
        }
        this._initializedChunks = chunkIndex + 1;
    }

    private void validatePointer(long ptr) {
        if (ptr >= this._position) {
            throw new IllegalArgumentException("Invalid pointer " + ptr + " does not in range [0," + this._position + ')');
        }
    }

    public byte getByte(long ptr) {
        this.validatePointer(ptr);
        int i = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[i];
        long j = this.offset(ptr);
        return this._UNSAFE.getByte(chunk, j);
    }

    public void putByte(long ptr, byte value) {
        this.validatePointer(ptr);
        int i = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[i];
        long j = this.offset(ptr);
        this._UNSAFE.putByte(chunk, j, value);
    }

    public int getInt(long ptr) {
        this.validatePointer(ptr);
        int i = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[i];
        long j = this.offset(ptr);
        return this._UNSAFE.getInt(chunk, j);
    }

    public void putInt(long ptr, int value) {
        this.validatePointer(ptr);
        int i = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[i];
        long j = this.offset(ptr);
        this._UNSAFE.putInt(chunk, j, value);
    }

    public short getShort(long ptr) {
        this.validatePointer(ptr);
        int i = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[i];
        long j = this.offset(ptr);
        return this._UNSAFE.getShort(chunk, j);
    }

    public void putShort(long ptr, short value) {
        this.validatePointer(ptr);
        int i = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[i];
        long j = this.offset(ptr);
        this._UNSAFE.putShort(chunk, j, value);
    }

    public char getChar(long ptr) {
        this.validatePointer(ptr);
        int i = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[i];
        long j = this.offset(ptr);
        return this._UNSAFE.getChar(chunk, j);
    }

    public void putChar(long ptr, char value) {
        this.validatePointer(ptr);
        int i = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[i];
        long j = this.offset(ptr);
        this._UNSAFE.putChar(chunk, j, value);
    }

    public long getLong(long ptr) {
        int i = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[i];
        long j = this.offset(ptr);
        return this._UNSAFE.getLong(chunk, j);
    }

    public void putLong(long ptr, long value) {
        this.validatePointer(ptr);
        int i = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[i];
        long j = this.offset(ptr);
        this._UNSAFE.putLong(chunk, j, value);
    }

    public float getFloat(long ptr) {
        this.validatePointer(ptr);
        int i = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[i];
        long j = this.offset(ptr);
        return this._UNSAFE.getFloat(chunk, j);
    }

    public void putFloat(long ptr, float value) {
        this.validatePointer(ptr);
        int i = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[i];
        long j = this.offset(ptr);
        this._UNSAFE.putFloat(chunk, j, value);
    }

    public double getDouble(long ptr) {
        this.validatePointer(ptr);
        int i = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[i];
        long j = this.offset(ptr);
        return this._UNSAFE.getDouble(chunk, j);
    }

    public void putDouble(long ptr, double value) {
        this.validatePointer(ptr);
        int i = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[i];
        long j = this.offset(ptr);
        this._UNSAFE.putDouble(chunk, j, value);
    }

    public void getFloats(long ptr, @Nonnull float[] values) {
        this.validatePointer(ptr);
        int len = values.length;
        if (len == 0) {
            throw new IllegalArgumentException("Cannot put empty array at " + ptr);
        }
        int chunkIdx = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[chunkIdx];
        long base = this.offset(ptr);
        for (int i = 0; i < len; ++i) {
            long offset = base + (long)(4 * i);
            this.validateOffset(offset);
            values[i] = this._UNSAFE.getFloat(chunk, offset);
        }
    }

    public void putFloats(long ptr, @Nonnull float[] values) {
        this.validatePointer(ptr);
        int len = values.length;
        if (len == 0) {
            throw new IllegalArgumentException("Cannot put empty array at " + ptr);
        }
        int chunkIdx = Primitives.castToInt(ptr / (long)this._chunkBytes);
        int[] chunk = this._chunks[chunkIdx];
        long base = this.offset(ptr);
        for (int i = 0; i < len; ++i) {
            long offset = base + (long)(4 * i);
            this.validateOffset(offset);
            this._UNSAFE.putFloat(chunk, offset, values[i]);
        }
    }

    private void validateOffset(long offset) {
        if (offset >= (long)this._chunkBytes) {
            throw new IndexOutOfBoundsException("Invalid offset " + offset + " not in range [0," + this._chunkBytes + ')');
        }
    }

    private long offset(long ptr) {
        long j = ptr % (long)this._chunkBytes;
        return (long)Unsafe.ARRAY_INT_BASE_OFFSET + j;
    }

    public String toString() {
        return "HeapBuffer [position=" + NumberUtils.formatNumber(this._position) + ", #allocatedObjects=" + NumberUtils.formatNumber(this._numAllocated) + ", #consumed=" + NumberUtils.prettySize(this.consumedBytes()) + ", #allocated=" + NumberUtils.prettySize(this._allocatedBytes) + ", #skipped=" + NumberUtils.prettySize(this._skippedBytes) + ", #chunks=" + NumberUtils.formatNumber(this._chunks.length) + ", #initializedChunks=" + NumberUtils.formatNumber(this._initializedChunks) + ", #chunkSize=" + NumberUtils.formatNumber(this._chunkSize) + ", #chunkBytes=" + NumberUtils.formatNumber(this._chunkBytes) + " bytes]";
    }

    public long consumedBytes() {
        return this._chunkBytes * this._initializedChunks;
    }

    public int getNumInitializedChunks() {
        return this._initializedChunks;
    }

    public int getNumChunks() {
        return this._chunks.length;
    }

    public long position() {
        return this._position;
    }

    public int getNumAllocated() {
        return this._numAllocated;
    }

    public long getAllocatedBytes() {
        return this._allocatedBytes;
    }

    public long getSkippedBytes() {
        return this._skippedBytes;
    }
}

